/* -*-c++-*- Copyright (C) 2018 Advanced Driver Information Technology.
 *
 * This library is open source and may be redistributed and/or modified under
 * the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or
 * (at your option) any later version.  The full license is in LICENSE file
 * included with this distribution, and on the openscenegraph.org website.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * OpenSceneGraph Public License for more details.
*/

#include <errno.h>
#include "WLWindowController.hpp"
#include "WLShellInterface.hpp"
#include <wayland-client.h>

using namespace WaylandBackend;

WLWindowController::WLWindowController(void* wlSurface,
                                       WLClientContext *wlClientCtx):
_wlSurface((struct wl_surface*)wlSurface),
_shellSurface(NULL),
_surfaceID(0),
_useSurfId(false)
{
    if (NULL != wlClientCtx)
    {
        _wlClientCtx = wlClientCtx;
        _wlShellIf = (WLShellInterface *)wlClientCtx->getActiveShell();
    }
}

int
WLWindowController::init(unsigned int surfaceId, unsigned int layerId,
                         unsigned int width, unsigned int height)
{
    int ret;
    ret = _wlShellIf->surfaceCreate(_wlSurface, surfaceId, layerId, width,
                                    height);
    if (ret >= 0)
    {
        _surfaceID = surfaceId;
        _useSurfId = true;
    }
    else if (-ENOTSUP == ret)
    {
        ret = _wlShellIf->surfaceCreate(_wlSurface, surfaceId, &_shellSurface);
    }
    return ret;
}

std::string&
WLWindowController::getShellName()
{
    return _wlShellIf->getShellName();
}

int
WLWindowController::Configure(int x, int y, unsigned int width,
                              unsigned int height)
{
    return _wlShellIf->surfaceConfigure(_surfaceID, _shellSurface, x, y,
                                        width, height);
}

int
WLWindowController::ConfigureSynchronous(int x, int y, unsigned int width,
                                         unsigned int height)
{
    return _wlShellIf->surfaceConfigureSynchronous(_surfaceID, _shellSurface,
                                                   x, y, width, height);
}

int
WLWindowController::setVisibility(bool visible)
{
    return _wlShellIf->surfaceSetVisbility(_surfaceID, _shellSurface,
                                           visible);
}

int
WLWindowController::setTitle(const char *title)
{
    int ret = -ENOTSUP;
    if (NULL != _shellSurface)
        ret = _wlShellIf->surfaceSetTitle(_shellSurface, title);
    return ret;
}


int
WLWindowController::setClass(const char *surfClass)
{
    int ret = -ENOTSUP;
    if (NULL != _shellSurface)
        ret = _wlShellIf->surfaceSetClass(_shellSurface, surfClass);
    return ret;
}

int
WLWindowController::setAsPopup(void *seat, uint32_t serial,
                               WLWindowController *parent,
                               int32_t x, int32_t y, uint32_t flags)
{
    int ret = -EINVAL;
    if ((NULL != _shellSurface) && (NULL != seat)
         && (NULL != parent))
    {
        ret = _wlShellIf->surfaceSetAsPopup(_shellSurface,
                (struct wl_seat*)seat,serial,
                (struct wl_surface*)parent->getWlSurface(),
                x, y, flags);
    }
    return ret;
}

int
WLWindowController::setMaximized(const char *outputName)
{
    int ret = -EINVAL;
    struct wl_output *output = (struct wl_output *)
                               _wlClientCtx->getWlOuput(outputName);
    if ((NULL != _shellSurface) && (NULL != output))
    {
        ret = _wlShellIf->surfaceSetMaximized(_shellSurface, output);
    }

    return ret;
}

int
WLWindowController::setFullScreen(uint32_t method,
                                  uint32_t framerate,
                                  const char *outputName)
{
    int ret = -EINVAL;
    struct wl_output *output = (struct wl_output *)
                               _wlClientCtx->getWlOuput(outputName);
    if ((NULL != _shellSurface) && (NULL != output))
    {
        ret = _wlShellIf->surfaceSetFullScreen(_shellSurface, method,
                                         framerate, output);
    }
    return ret;
}

int
WLWindowController::setTransient(WLWindowController *parent,
                                int32_t x, int32_t y, uint32_t flags)
{
    int ret = -EINVAL;
    if ((NULL != _shellSurface) && (NULL != parent))
    {
        ret = _wlShellIf->surfaceSetTransient(_shellSurface,
                            (struct wl_surface*)parent->getWlSurface(),
                             x, y, flags);
    }
    return ret;
}

int
WLWindowController::setTopLevel()
{
    int ret = -EINVAL;
    if (NULL != _shellSurface)
    {
        ret = _wlShellIf->surfaceSetTopLevel(_shellSurface);
    }
    return ret;
}

int
WLWindowController::resize(void *seat, uint32_t serial, uint32_t edges)
{
    int ret = -EINVAL;
    if ((NULL != _shellSurface) && (NULL != seat))
    {
        ret = _wlShellIf->surfaceResize(_shellSurface, (struct wl_seat*)seat,
                                        serial, edges);
    }
    return ret;
}

int
WLWindowController::move(void *seat, uint32_t serial)
{
    int ret = -EINVAL;
    if ((NULL != _shellSurface) && (NULL != seat))
    {
        ret = _wlShellIf->surfaceMove(_shellSurface, (struct wl_seat*)seat,
                                        serial);
    }
    return ret;
}


void*
WLWindowController::getWlSurface()
{
    return (void*)_wlSurface;
}

WLWindowController::~WLWindowController()
{
    if (NULL != _shellSurface)
    {
        _wlShellIf->surfaceDestroy(_shellSurface);
    }
    if (_useSurfId)
    {
        _wlShellIf->surfaceDestroy(_surfaceID);
    }
}
